volatile unsigned long long __coreTime = 0;

_INTERRUPT_JUMP (_TIMER0_OVF)
{
	__coreTime++;
}

void Core::startGlobalInterrupt()
{
	_SREG |= 0b10000000;
}

unsigned long long Core::s()
{
	unsigned char tcnt0Save = 0;
	unsigned long long s = 0;
	
	_SREG &= ~0b10000000;
	
	tcnt0Save = _TCNT0;
	
	if ((_TIFR0 & 0b00000001) == 0b00000001)
	{
		tcnt0Save = _TCNT0;
		_TIFR0 |= 0b00000001;
		
		__coreTime++;
	}
	
	s = ((__coreTime * 256ull) + (unsigned long long)(tcnt0Save)) / 2000000ull;
	
	_SREG |= 0b10000000;
	
	return s;
}

unsigned long long Core::ms()
{
	unsigned char tcnt0Save = 0;
	unsigned long long ms = 0;
	
	_SREG &= ~0b10000000;
	
	tcnt0Save = _TCNT0;
	
	if ((_TIFR0 & 0b00000001) == 0b00000001)
	{
		tcnt0Save = _TCNT0;
		_TIFR0 |= 0b00000001;
		
		__coreTime++;
	}
	
	ms = ((__coreTime * 256ull) + (unsigned long long)(tcnt0Save)) / 2000ull;
	
	_SREG |= 0b10000000;
	
	return ms;
}

unsigned long long Core::us()
{
	unsigned char tcnt0Save = 0;
	unsigned long long us = 0;
	
	_SREG &= ~0b10000000;
	
	tcnt0Save = _TCNT0;
	
	if ((_TIFR0 & 0b00000001) == 0b00000001)
	{
		tcnt0Save = _TCNT0;
		_TIFR0 |= 0b00000001;
		
		__coreTime++;
	}
	
	us = ((__coreTime * 256ull) + (unsigned long long)(tcnt0Save)) / 2ull;
	
	_SREG |= 0b10000000;
	
	return us;
}

volatile unsigned char *Core::pinToRegisterPort (const unsigned char PIN)
{
	volatile unsigned char *registerPort = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN >= 1 && PIN <= 8)
	{
		registerPort = &_PORTD;
	}
	else if (PIN >= 9 && PIN <= 14)
	{
		registerPort = &_PORTB;
	}
	else if (PIN >= 15 && PIN <= 20)
	{
		registerPort = &_PORTC;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN >= 1 && PIN <= 8)
	{
		registerPort = &_PORTB;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerPort = &_PORTD;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerPort = &_PORTC;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerPort = &_PORTA;
	}
	#endif
	
	return registerPort;
}

volatile unsigned char *Core::pinToRegisterDdr (const unsigned char PIN)
{
	volatile unsigned char *registerDdr = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN >= 1 && PIN <= 8)
	{
		registerDdr = &_DDRD;
	}
	else if (PIN >= 9 && PIN <= 14)
	{
		registerDdr = &_DDRB;
	}
	else if (PIN >= 15 && PIN <= 20)
	{
		registerDdr = &_DDRC;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN >= 1 && PIN <= 8)
	{
		registerDdr = &_DDRB;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerDdr = &_DDRD;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerDdr = &_DDRC;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerDdr = &_DDRA;
	}
	#endif
	
	return registerDdr;
}

volatile unsigned char *Core::pinToRegisterPin (const unsigned char PIN)
{
	volatile unsigned char *registerPin = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN >= 1 && PIN <= 8)
	{
		registerPin = &_PIND;
	}
	else if (PIN >= 9 && PIN <= 14)
	{
		registerPin = &_PINB;
	}
	else if (PIN >= 15 && PIN <= 20)
	{
		registerPin = &_PINC;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN >= 1 && PIN <= 8)
	{
		registerPin = &_PINB;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerPin = &_PIND;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerPin = &_PINC;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerPin = &_PINA;
	}
	#endif
	
	return registerPin;
}

unsigned char Core::pinToBytePort (const unsigned char PIN)
{
	unsigned char bytePort = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN == 1 || PIN == 9 || PIN == 15)
	{
		bytePort = 0b00000001;
	}
	else if (PIN == 2 || PIN == 10 || PIN == 16)
	{
		bytePort = 0b00000010;
	}
	else if (PIN == 3 || PIN == 11 || PIN == 17)
	{
		bytePort = 0b00000100;
	}
	else if (PIN == 4 || PIN == 12 || PIN == 18)
	{
		bytePort = 0b00001000;
	}
	else if (PIN == 5 || PIN == 13 || PIN == 19)
	{
		bytePort = 0b00010000;
	}
	else if (PIN == 6 || PIN == 14 || PIN == 20)
	{
		bytePort = 0b00100000;
	}
	else if (PIN == 7)
	{
		bytePort = 0b01000000;
	}
	else if (PIN == 8)
	{
		bytePort = 0b10000000;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN == 1 || PIN == 9 || PIN == 17 || PIN == 32)
	{
		bytePort = 0b00000001;
	}
	else if (PIN == 2 || PIN == 10 || PIN == 18 || PIN == 31)
	{
		bytePort = 0b00000010;
	}
	else if (PIN == 3 || PIN == 11 || PIN == 19 || PIN == 30)
	{
		bytePort = 0b00000100;
	}
	else if (PIN == 4 || PIN == 12 || PIN == 20 || PIN == 29)
	{
		bytePort = 0b00001000;
	}
	else if (PIN == 5 || PIN == 13 || PIN == 21 || PIN == 28)
	{
		bytePort = 0b00010000;
	}
	else if (PIN == 6 || PIN == 14 || PIN == 22 || PIN == 27)
	{
		bytePort = 0b00100000;
	}
	else if (PIN == 7 || PIN == 15 || PIN == 23 || PIN == 26)
	{
		bytePort = 0b01000000;
	}
	else if (PIN == 8 || PIN == 16 || PIN == 24 || PIN == 25)
	{
		bytePort = 0b10000000;
	}
	#endif
	
	return bytePort;
}

unsigned char Core::pinToInterrupt (const unsigned char PIN)
{
	unsigned char interrupt = 255;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN == 3)
	{
		interrupt = 0;
	}
	else if (PIN == 4)
	{
		interrupt = 1;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN == 3)
	{
		interrupt = 2;
	}
	else if (PIN == 11)
	{
		interrupt = 0;
	}
	else if (PIN == 12)
	{
		interrupt = 1;
	}
	#endif
	
	return interrupt;
}

volatile unsigned char *Core::pinToRegisterPcmsk (const unsigned char PIN)
{
	volatile unsigned char *registerPcmsk = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN >= 1 && PIN <= 8)
	{
		registerPcmsk = &_PCMSK2;
	}
	else if (PIN >= 9 && PIN <= 14)
	{
		registerPcmsk = &_PCMSK0;
	}
	else if (PIN >= 15 && PIN <= 20)
	{
		registerPcmsk = &_PCMSK1;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN >= 1 && PIN <= 8)
	{
		registerPcmsk = &_PCMSK1;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		registerPcmsk = &_PCMSK3;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		registerPcmsk = &_PCMSK2;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		registerPcmsk = &_PCMSK0;
	}
	#endif
	
	return registerPcmsk;
}

unsigned char Core::pinToBytePcie (const unsigned char PIN)
{
	unsigned char bytePcie = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN >= 1 && PIN <= 8)
	{
		bytePcie = 0b00000100;
	}
	else if (PIN >= 9 && PIN <= 14)
	{
		bytePcie = 0b00000001;
	}
	else if (PIN >= 15 && PIN <= 20)
	{
		bytePcie = 0b00000010;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	if (PIN >= 1 && PIN <= 8)
	{
		bytePcie = 0b00000010;
	}
	else if (PIN >= 9 && PIN <= 16)
	{
		bytePcie = 0b00001000;
	}
	else if (PIN >= 17 && PIN <= 24)
	{
		bytePcie = 0b00000100;
	}
	else if (PIN >= 25 && PIN <= 32)
	{
		bytePcie = 0b00000001;
	}
	#endif
	
	return bytePcie;
}

bool Core::pinIsChannelTimerPwm (const unsigned char PIN)
{
	bool is = false;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN == 10 || PIN == 11)
	{
		is = true;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P
	if (PIN == 13 || PIN == 14)
	{
		is = true;
	}
	#elif defined _ATMEGA1284P
	if (PIN == 7 || PIN == 8 || PIN == 13 || PIN == 14)
	{
		is = true;
	}
	#endif
	
	return is;
}

volatile unsigned char *Core::pinToRegisterTccr (const unsigned char PIN)
{
	volatile unsigned char *registerTccr = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN == 10 || PIN == 11)
	{
		registerTccr = &_TCCR1A;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P
	if (PIN == 13 || PIN == 14)
	{
		registerTccr = &_TCCR1A;
	}
	#elif defined _ATMEGA1284P
	if (PIN == 7 || PIN == 8)
	{
		registerTccr = &_TCCR3A;
	}
	else if (PIN == 13 || PIN == 14)
	{
		registerTccr = &_TCCR1A;
	}
	#endif
	
	return registerTccr;
}

volatile unsigned int *Core::pinToRegisterOcr (const unsigned char PIN)
{
	volatile unsigned int *registerOcr = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN == 10)
	{
		registerOcr = &_OCR1ALH;
	}
	else if (PIN == 11)
	{
		registerOcr = &_OCR1BLH;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P
	if (PIN == 13)
	{
		registerOcr = &_OCR1BLH;
	}
	else if (PIN == 14)
	{
		registerOcr = &_OCR1ALH;
	}
	#elif defined _ATMEGA1284P
	if (PIN == 7)
	{
		registerOcr = &_OCR3ALH;
	}
	else if (PIN == 8)
	{
		registerOcr = &_OCR3BLH;
	}
	else if (PIN == 13)
	{
		registerOcr = &_OCR1BLH;
	}
	else if (PIN == 14)
	{
		registerOcr = &_OCR1ALH;
	}
	#endif
	
	return registerOcr;
}

unsigned char Core::pinToByteCom (const unsigned char PIN)
{
	unsigned char byteCom = 0;
	
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	if (PIN == 10)
	{
		byteCom = 0b10000000;
	}
	else if (PIN == 11)
	{
		byteCom = 0b00100000;
	}
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P
	if (PIN == 13)
	{
		byteCom = 0b00100000;
	}
	else if (PIN == 14)
	{
		byteCom = 0b10000000;
	}
	#elif defined _ATMEGA1284P
	if (PIN == 7 || PIN == 14)
	{
		byteCom = 0b10000000;
	}
	else if (PIN == 8 || PIN == 13)
	{
		byteCom = 0b00100000;
	}
	#endif
	
	return byteCom;
}

unsigned char Core::pinToByteAdmux (const unsigned char PIN)
{
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	return 0b01000000 | (PIN - 15);
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	return 0b01000000 | (32 - PIN);
	#endif
}

void Core::startInterruptTimerCounter()
{
	_TCCR0B = 0b00000010;
	_TIMSK0 = 0b00000001;
}

void Core::modeTimerPwm()
{
	_TCCR1A = 0b00000010;
	_TCCR1B = 0b00011000;
	#if defined _ATMEGA1284P
	_TCCR3A = 0b00000010;
	_TCCR3B = 0b00011000;
	#endif
}

unsigned int Core::frequencyToPrescalerTimerPwm (const float FREQUENCY)
{
	unsigned int prescaler = 1024;
	
	if (((16000000.0 / 1.0) / FREQUENCY) - 0.5 <= 65535)
	{
		prescaler = 1;
	}
	else if (((16000000.0 / 8.0) / FREQUENCY) - 0.5 <= 65535)
	{
		prescaler = 8;
	}
	else if (((16000000.0 / 64.0) / FREQUENCY) - 0.5 <= 65535)
	{
		prescaler = 64;
	}
	else if (((16000000.0 / 256.0) / FREQUENCY) - 0.5 <= 65535)
	{
		prescaler = 256;
	}
	
	return prescaler;
}

void Core::prescalerTimerPwm (const unsigned int PRESCALER)
{
	if (PRESCALER == 1)
	{
		_TCCR1B |= 0b00000001;
		#if defined _ATMEGA1284P
		_TCCR3B |= 0b00000001;
		#endif
	}
	else if (PRESCALER == 8)
	{
		_TCCR1B |= 0b00000010;
		#if defined _ATMEGA1284P
		_TCCR3B |= 0b00000010;
		#endif
	}
	else if (PRESCALER == 64)
	{
		_TCCR1B |= 0b00000011;
		#if defined _ATMEGA1284P
		_TCCR3B |= 0b00000011;
		#endif
	}
	else if (PRESCALER == 256)
	{
		_TCCR1B |= 0b00000100;
		#if defined _ATMEGA1284P
		_TCCR3B |= 0b00000100;
		#endif
	}
	else if (PRESCALER == 1024)
	{
		_TCCR1B |= 0b00000101;
		#if defined _ATMEGA1284P
		_TCCR3B |= 0b00000101;
		#endif
	}
}

void Core::inputCaptureTimerPwm (const unsigned int VALUE)
{
	_ICR1LH = VALUE;
	#if defined _ATMEGA1284P
	_ICR3LH = VALUE;
	#endif
}

void Core::outputCompareTimerPwm (const unsigned int VALUE)
{
	_OCR1ALH = VALUE;
}

void Core::startInterruptTimerPwm()
{
	_TIMSK1 = 0b00000011;
}

void Core::stopInterruptTimerPwm()
{
	_TIMSK1 = 0b00000000;
}

void Core::modeTimerSound()
{
	_TCCR2A = 0b00000010;
}

unsigned int Core::frequencyToPrescalerTimerSound (const float FREQUENCY)
{
	unsigned int prescaler = 1024;
	
	if (((16000000.0 / 1.0) / FREQUENCY) - 0.5 <= 255)
	{
		prescaler = 1;
	}
	else if (((16000000.0 / 8.0) / FREQUENCY) - 0.5 <= 255)
	{
		prescaler = 8;
	}
	else if (((16000000.0 / 32.0) / FREQUENCY) - 0.5 <= 255)
	{
		prescaler = 32;
	}
	else if (((16000000.0 / 64.0) / FREQUENCY) - 0.5 <= 255)
	{
		prescaler = 64;
	}
	else if (((16000000.0 / 128.0) / FREQUENCY) - 0.5 <= 255)
	{
		prescaler = 128;
	}
	else if (((16000000.0 / 256.0) / FREQUENCY) - 0.5 <= 255)
	{
		prescaler = 256;
	}
	
	return prescaler;
}

void Core::prescalerTimerSound (const unsigned int PRESCALER)
{
	if (PRESCALER == 1)
	{
		_TCCR2B = 0b00000001;
	}
	else if (PRESCALER == 8)
	{
		_TCCR2B = 0b00000010;
	}
	else if (PRESCALER == 32)
	{
		_TCCR2B = 0b00000011;
	}
	else if (PRESCALER == 64)
	{
		_TCCR2B = 0b00000100;
	}
	else if (PRESCALER == 128)
	{
		_TCCR2B = 0b00000101;
	}
	else if (PRESCALER == 256)
	{
		_TCCR2B = 0b00000110;
	}
	else if (PRESCALER == 1024)
	{
		_TCCR2B = 0b00000111;
	}
}

void Core::outputCompareTimerSound (const unsigned char VALUE)
{
	_OCR2A = VALUE;
}

void Core::startInterruptTimerSound()
{
	_TIMSK2 = 0b00000010;
}

void Core::stopInterruptTimerSound()
{
	_TIMSK2 = 0b00000000;
}

void Core::startInterrupt (const unsigned char INTERRUPT, const bool RISING, const bool FALLING)
{
	if (INTERRUPT == 0)
	{
		if (RISING == true && FALLING == false)
		{
			_EICRA |= 0b00000011;
		}
		else if (RISING == false && FALLING == true)
		{
			_EICRA |= 0b00000010;
		}
		else
		{
			_EICRA |= 0b00000001;
		}
		
		_EIMSK |= 0b00000001;
	}
	else if (INTERRUPT == 1)
	{
		if (RISING == true && FALLING == false)
		{
			_EICRA |= 0b00001100;
		}
		else if (RISING == false && FALLING == true)
		{
			_EICRA |= 0b00001000;
		}
		else
		{
			_EICRA |= 0b00000100;
		}
		
		_EIMSK |= 0b00000010;
	}
	#if defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	else if (INTERRUPT == 2)
	{
		if (RISING == true && FALLING == false)
		{
			_EICRA |= 0b00110000;
		}
		else if (RISING == false && FALLING == true)
		{
			_EICRA |= 0b00100000;
		}
		else
		{
			_EICRA |= 0b00010000;
		}
		
		_EIMSK |= 0b00000100;
	}
	#endif
}

void Core::stopInterrupt (const unsigned char INTERRUPT)
{
	if (INTERRUPT == 0)
	{
		_EIMSK &= ~0b00000001;
	}
	else if (INTERRUPT == 1)
	{
		_EIMSK &= ~0b00000010;
	}
	#if defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	else if (INTERRUPT == 2)
	{
		_EIMSK &= ~0b00000100;
	}
	#endif
}

void Core::startPinChangeInterrupt (const unsigned char BYTE_PCIE)
{
	_PCICR = BYTE_PCIE;
}

void Core::stopPinChangeInterrupt()
{
	_PCICR = 0b00000000;
}

unsigned char Core::digitToByte (const unsigned char DIGIT, const bool COMMA)
{
	unsigned char byte = 0;
	
	if (DIGIT == 0)
	{
		byte = 0b01111110;
	}
	else if (DIGIT == 1)
	{
		byte = 0b00110000;
	}
	else if (DIGIT == 2)
	{
		byte = 0b01101101;
	}
	else if (DIGIT == 3)
	{
		byte = 0b01111001;
	}
	else if (DIGIT == 4)
	{
		byte = 0b00110011;
	}
	else if (DIGIT == 5)
	{
		byte = 0b01011011;
	}
	else if (DIGIT == 6)
	{
		byte = 0b01011111;
	}
	else if (DIGIT == 7)
	{
		byte = 0b01110000;
	}
	else if (DIGIT == 8)
	{
		byte = 0b01111111;
	}
	else if (DIGIT == 9)
	{
		byte = 0b01111011;
	}
	
	if (COMMA == true)
	{
		byte |= 0b10000000;
	}
	
	return byte;
}

unsigned char Core::characterToByte (const char CHARACTER)
{
	unsigned char byte = 0;
	
	if (CHARACTER == 0b01110001)
	{
		byte = 0b01111110;
	}
	else if (CHARACTER == 0b01110111)
	{
		byte = 0b00111110;
	}
	else if (CHARACTER == 0b01100101)
	{
		byte = 0b01001111;
	}
	else if (CHARACTER == 0b01110010)
	{
		byte = 0b01000110;
	}
	else if (CHARACTER == 0b01110100)
	{
		byte = 0b00000111;
	}
	else if (CHARACTER == 0b01111001)
	{
		byte = 0b00110011;
	}
	else if (CHARACTER == 0b01110101)
	{
		byte = 0b00111110;
	}
	else if (CHARACTER == 0b01101001)
	{
		byte = 0b00110000;
	}
	else if (CHARACTER == 0b01101111)
	{
		byte = 0b01111110;
	}
	else if (CHARACTER == 0b01110000)
	{
		byte = 0b01100111;
	}
	else if (CHARACTER == 0b01100001)
	{
		byte = 0b01110111;
	}
	else if (CHARACTER == 0b01110011)
	{
		byte = 0b01011011;
	}
	else if (CHARACTER == 0b01100100)
	{
		byte = 0b01111000;
	}
	else if (CHARACTER == 0b01100110)
	{
		byte = 0b01000111;
	}
	else if (CHARACTER == 0b01100111)
	{
		byte = 0b01011110;
	}
	else if (CHARACTER == 0b01101000)
	{
		byte = 0b00110111;
	}
	else if (CHARACTER == 0b01101010)
	{
		byte = 0b00111000;
	}
	else if (CHARACTER == 0b01101011)
	{
		byte = 0b00110111;
	}
	else if (CHARACTER == 0b01101100)
	{
		byte = 0b00001110;
	}
	else if (CHARACTER == 0b01111010)
	{
		byte = 0b01101101;
	}
	else if (CHARACTER == 0b01111000)
	{
		byte = 0b00110111;
	}
	else if (CHARACTER == 0b01100011)
	{
		byte = 0b01001110;
	}
	else if (CHARACTER == 0b01110110)
	{
		byte = 0b00111110;
	}
	else if (CHARACTER == 0b01100010)
	{
		byte = 0b01111111;
	}
	else if (CHARACTER == 0b01101110)
	{
		byte = 0b01110110;
	}
	else if (CHARACTER == 0b01101101)
	{
		byte = 0b01110110;
	}
	else if (CHARACTER == 0b00101110)
	{
		byte = 0b10000000;
	}
	else if (CHARACTER == 0b00101101)
	{
		byte = 0b00000001;
	}
	else if (CHARACTER == 0b00111101)
	{
		byte = 0b00001001;
	}
	else if (CHARACTER == 0b00110000)
	{
		byte = 0b01111110;
	}
	else if (CHARACTER == 0b00110001)
	{
		byte = 0b00110000;
	}
	else if (CHARACTER == 0b00110010)
	{
		byte = 0b01101101;
	}
	else if (CHARACTER == 0b00110011)
	{
		byte = 0b01111001;
	}
	else if (CHARACTER == 0b00110100)
	{
		byte = 0b00110011;
	}
	else if (CHARACTER == 0b00110101)
	{
		byte = 0b01011011;
	}
	else if (CHARACTER == 0b00110110)
	{
		byte = 0b01011111;
	}
	else if (CHARACTER == 0b00110111)
	{
		byte = 0b01110000;
	}
	else if (CHARACTER == 0b00111000)
	{
		byte = 0b01111111;
	}
	else if (CHARACTER == 0b00111001)
	{
		byte = 0b01111011;
	}
	
	return byte;
}

void Core::startSpi()
{
	#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
	_DDRB |= 0b00101100;
	#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
	_DDRB |= 0b10110000;
	#endif
	_SPCR = 0b01010000;
}

unsigned char Core::deviceSpi (const unsigned char PIN_SS)
{
	_countSpiDevice [PIN_SS - 1]++;
	
	return _countSpiDevice [PIN_SS - 1];
}

signed long Core::readSpi (const unsigned char ADDRESS, const signed char LENGTH)
{
	unsigned char shift = 0;
	signed long data = 0;
	
	_SPDR = ADDRESS;
	
	while ((_SPSR & 0b10000000) != 0b10000000)
	{
	}
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR = 0b11111111;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
			
			if (LENGTH == -8)
			{
				data |= (signed char)(_SPDR) << shift;
			}
			else if (LENGTH == -16)
			{
				data |= (signed int)(_SPDR) << shift;
			}
			else if (LENGTH == -32)
			{
				data |= (signed long)(_SPDR) << shift;
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR = 0b11111111;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
			
			if (LENGTH == 8)
			{
				data |= (signed char)(_SPDR) << (shift - 8);
			}
			else if (LENGTH == 16)
			{
				data |= (signed int)(_SPDR) << (shift - 8);
			}
			else if (LENGTH == 32)
			{
				data |= (signed long)(_SPDR) << (shift - 8);
			}
		}
	}
	
	return data;
}

signed long Core::readSpiPush (const signed char LENGTH)
{
	unsigned char shift = 0;
	signed long data = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR = 0b11111111;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
			
			if (LENGTH == -8)
			{
				data |= (signed char)(_SPDR) << shift;
			}
			else if (LENGTH == -16)
			{
				data |= (signed int)(_SPDR) << shift;
			}
			else if (LENGTH == -32)
			{
				data |= (signed long)(_SPDR) << shift;
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR = 0b11111111;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
			
			if (LENGTH == 8)
			{
				data |= (signed char)(_SPDR) << (shift - 8);
			}
			else if (LENGTH == 16)
			{
				data |= (signed int)(_SPDR) << (shift - 8);
			}
			else if (LENGTH == 32)
			{
				data |= (signed long)(_SPDR) << (shift - 8);
			}
		}
	}
	
	return data;
}

void Core::writeSpi (const unsigned char ADDRESS, const signed long DATA, const signed char LENGTH)
{
	unsigned char shift = 0;
	
	_SPDR = ADDRESS;
	
	while ((_SPSR & 0b10000000) != 0b10000000)
	{
	}
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR = DATA >> shift;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR = DATA >> (shift - 8);
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
		}
	}
}

void Core::writeSpiPush (const signed long DATA, const signed char LENGTH)
{
	unsigned char shift = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR = DATA >> shift;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR = DATA >> (shift - 8);
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
		}
	}
}

void Core::writeSpiCascaded (const unsigned char PIN_SS, const unsigned char DEVICE, const unsigned char ADDRESS, const signed long DATA, const signed char LENGTH)
{
	unsigned char offset = 2;
	const unsigned char COUNT_SPI_DEVICE = _countSpiDevice [PIN_SS - 1];
	unsigned char shift = 0;
	
	for (offset = 2; offset < (COUNT_SPI_DEVICE - (DEVICE - 1)) * 2; offset++)
	{
		_SPDR = 0b00000000;
		
		while ((_SPSR & 0b10000000) != 0b10000000)
		{
		}
	}
	
	_SPDR = ADDRESS;
	
	while ((_SPSR & 0b10000000) != 0b10000000)
	{
	}
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_SPDR = DATA >> shift;
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_SPDR = DATA >> (shift - 8);
			
			while ((_SPSR & 0b10000000) != 0b10000000)
			{
			}
		}
	}
	
	for (offset = 2; offset < DEVICE * 2; offset++)
	{
		_SPDR = 0b00000000;
		
		while ((_SPSR & 0b10000000) != 0b10000000)
		{
		}
	}
}

void Core::startUsart (const unsigned long FREQUENCY)
{
	_UCSR0A = 0b00000010;
	_UCSR0B = 0b10011000;
	_UCSR0C = 0b00000110;
	_UBRR0LH = (16000000.0 / (8.0 * (float)(FREQUENCY))) - 0.5;
}

signed long Core::readUsart (const signed char LENGTH)
{
	unsigned char shift = 0;
	signed long data = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			while ((_UCSR0A & 0b10000000) != 0b10000000)
			{
			}
			
			if (LENGTH == -8)
			{
				data |= (signed char)(_UDR0) << shift;
			}
			else if (LENGTH == -16)
			{
				data |= (signed int)(_UDR0) << shift;
			}
			else if (LENGTH == -32)
			{
				data |= (signed long)(_UDR0) << shift;
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			while ((_UCSR0A & 0b10000000) != 0b10000000)
			{
			}
			
			if (LENGTH == 8)
			{
				data |= (signed char)(_UDR0) << (shift - 8);
			}
			else if (LENGTH == 16)
			{
				data |= (signed int)(_UDR0) << (shift - 8);
			}
			else if (LENGTH == 32)
			{
				data |= (signed long)(_UDR0) << (shift - 8);
			}
		}
	}
	
	return data;
}

void Core::writeUsart (const signed long DATA, const signed char LENGTH)
{
	unsigned char shift = 0;
	
	if (LENGTH < 0)
	{
		for (shift = 0; shift != -LENGTH; shift += 8)
		{
			_UDR0 = DATA >> shift;
			
			while ((_UCSR0A & 0b00100000) != 0b00100000)
			{
			}
		}
	}
	else
	{
		for (shift = LENGTH; shift != 0; shift -= 8)
		{
			_UDR0 = DATA >> (shift - 8);
			
			while ((_UCSR0A & 0b00100000) != 0b00100000)
			{
			}
		}
	}
}

void Core::stopUsart()
{
	_UCSR0A = 0b00000000;
	_UCSR0B = 0b00000000;
	_UCSR0C = 0b00000000;
	_UBRR0L = 0b00000000;
}

void Core::startTwi()
{
	_TWBR = 0b00001100;
}

bool Core::scanTwi (const unsigned char DEVICE)
{
	unsigned int error = 0;
	unsigned char retrySequence = 0;
	unsigned char check = 0;
	bool foundDevice = false;
	
	while (error == 0 && retrySequence < 16)
	{
		_TWCR = 0b10100100;
		
		for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
		{
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWDR = DEVICE << 1;
			_TWCR = 0b10000100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWCR = 0b10010100;
			
			for (error = 512; (_TWCR & 0b00010000) != 0b00010000 && error > 0; error--)
			{
			}
		}
		
		if (error == 0)
		{
			_TWCR = 0b00000000;
			retrySequence++;
		}
	}
	
	if (retrySequence < 16)
	{
		foundDevice = true;
	}
	
	return foundDevice;
}

signed long Core::readTwi (const unsigned char DEVICE, const unsigned char ADDRESS, const signed char LENGTH)
{
	unsigned int error = 0;
	unsigned char retrySequence = 0;
	unsigned char check = 0;
	unsigned char shift = 0;
	signed long data = 0;
	
	while (error == 0 && retrySequence < 16)
	{
		_TWCR = 0b10100100;
		
		for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
		{
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWDR = DEVICE << 1;
			_TWCR = 0b10000100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWDR = ADDRESS;
			_TWCR = 0b10000100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00101000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWCR = 0b10100100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWDR = (DEVICE << 1) + 0b00000001;
			_TWCR = 0b10000100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			if (LENGTH < 0)
			{
				for (shift = 0; shift != -LENGTH && error != 0; shift += 8)
				{
					if (shift == -LENGTH - 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
					{
					}
					
					if (LENGTH == -8)
					{
						data |= (signed char)(_TWDR) << shift;
					}
					else if (LENGTH == -16)
					{
						data |= (signed int)(_TWDR) << shift;
					}
					else if (LENGTH == -32)
					{
						data |= (signed long)(_TWDR) << shift;
					}
				}
			}
			else
			{
				for (shift = LENGTH; shift != 0 && error != 0; shift -= 8)
				{
					if (shift == 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
					{
					}
					
					if (LENGTH == 8)
					{
						data |= (signed char)(_TWDR) << (shift - 8);
					}
					else if (LENGTH == 16)
					{
						data |= (signed int)(_TWDR) << (shift - 8);
					}
					else if (LENGTH == 32)
					{
						data |= (signed long)(_TWDR) << (shift - 8);
					}
				}
			}
		}
		
		if (error != 0)
		{
			_TWCR = 0b10010100;
			
			for (error = 512; (_TWCR & 0b00010000) != 0b00010000 && error > 0; error--)
			{
			}
		}
		
		if (error == 0)
		{
			_TWCR = 0b00000000;
			data = 0;
			retrySequence++;
		}
	}
	
	return data;
}

void Core::writeTwi (const unsigned char DEVICE, const unsigned char ADDRESS, const signed long DATA, const signed char LENGTH)
{
	unsigned int error = 0;
	unsigned char retrySequence = 0;
	unsigned char check = 0;
	unsigned char shift = 0;
	
	while (error == 0 && retrySequence < 16)
	{
		_TWCR = 0b00000000;
		_TWCR = 0b10100100;
		
		for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
		{
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00001000 && check != 0b00010000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWDR = DEVICE << 1;
			_TWCR = 0b10000100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00011000 && check != 0b01000000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWDR = ADDRESS;
			_TWCR = 0b10000100;
			
			for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
			{
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00101000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			if (LENGTH < 0)
			{
				for (shift = 0; shift != -LENGTH && error != 0; shift += 8)
				{
					_TWDR = DATA >> shift;
					
					if (shift == -LENGTH - 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
					{
					}
				}
			}
			else
			{
				for (shift = LENGTH; shift != 0 && error != 0; shift -= 8)
				{
					_TWDR = DATA >> (shift - 8);
					
					if (shift == 8)
					{
						_TWCR = 0b10000100;
					}
					else
					{
						_TWCR = 0b11000100;
					}
					
					for (error = 512; (_TWCR & 0b10000000) != 0b10000000 && error > 0; error--)
					{
					}
				}
			}
		}
		
		if (error != 0)
		{
			check = _TWSR & 0b11111000;
			
			if (check != 0b00101000)
			{
				error = 0;
			}
		}
		
		if (error != 0)
		{
			_TWCR = 0b10010100;
			
			for (error = 512; (_TWCR & 0b00010000) != 0b00010000 && error > 0; error--)
			{
			}
		}
		
		if (error == 0)
		{
			_TWCR = 0b00000000;
			retrySequence++;
		}
	}
}

void Core::startAnalog()
{
	_ADCSRA = 0b10000101;
}

unsigned int Core::readAnalog (const unsigned char BYTE_ADMUX)
{
	_ADMUX = BYTE_ADMUX;
	_ADCSRA |= 0b01000000;
	
	while ((_ADCSRA & 0b01000000) != 0b00000000)
	{
	}
	
	return _ADCLH;
}

#if defined _ATMEGA48P
unsigned char Core::readEeprom (const unsigned char ADDRESS)
#elif defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P || defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
unsigned char Core::readEeprom (const unsigned int ADDRESS)
#endif
{
	const unsigned char SREG_SAVE = _SREG;
	
	_SREG &= ~0b10000000;
	
	_EEARLH = ADDRESS;
	_EECR |= 0b00000001;
	
	while ((_EECR & 0b00000010) != 0b00000000)
	{
	}
	
	_SREG = SREG_SAVE;
	
	return _EEDR;
}

#if defined _ATMEGA48P
void Core::writeEeprom (const unsigned char ADDRESS, const unsigned char DATA)
#elif defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P || defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
void Core::writeEeprom (const unsigned int ADDRESS, const unsigned char DATA)
#endif
{
	const unsigned char SREG_SAVE = _SREG;
	
	_SREG &= ~0b10000000;
	
	_EEARLH = ADDRESS;
	_EEDR = DATA;
	_EECR |= 0b00000100;
	_EECR |= 0b00000010;
	
	while ((_EECR & 0b00000010) != 0b00000000)
	{
	}
	
	_SREG = SREG_SAVE;
}

unsigned long Core::crc32 (const signed long DATA)
{
	unsigned long seed = 0;
	unsigned char shift = 0;
	float artificialNoise = 1;
	unsigned long value = 0;
	
	if (DATA >= 0)
	{
		seed = DATA;
	}
	else
	{
		seed = DATA + 2147483648ul;
	}
	
	for (shift = 0; shift < 32; shift++)
	{
		artificialNoise = (artificialNoise + (float)(seed)) * 0.9;
		
		if (seed < 4294967295)
		{
			seed++;
		}
		else
		{
			seed = 0;
		}
		
		value |= (unsigned long)(artificialNoise & 0b00000000000000000000000000000001) << shift;
	}
	
	return value;
}

void Core::power (const bool POWER)
{
	if (POWER == false)
	{
		#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
		_PRR = 0b11101111;
		#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
		_PRR0 = 0b11111111;
		#endif
		_SMCR = 0b00000101;
		
		_MCU_SLEEP();
	}
	else
	{
		_SMCR = 0b00000000;
		#if defined _ATMEGA48P || defined _ATMEGA88P || defined _ATMEGA168P || defined _ATMEGA328P
		_PRR = 0b00000000;
		#elif defined _ATMEGA164P || defined _ATMEGA324P || defined _ATMEGA644P || defined _ATMEGA1284P
		_PRR0 = 0b00000000;
		#endif
	}
}

signed long Core::floor (const float VALUE)
{
	signed long value = 0;
	
	if (VALUE <= 0)
	{
		value = VALUE - 1.0;
	}
	else
	{
		value = VALUE;
	}
	
	return value;
}

signed long Core::round (const float VALUE)
{
	signed long value = 0;
	
	if (VALUE <= 0)
	{
		value = VALUE - 0.5;
	}
	else
	{
		value = VALUE + 0.5;
	}
	
	return value;
}

signed long Core::ceil (const float VALUE)
{
	signed long value = 0;
	
	if (VALUE <= 0)
	{
		value = VALUE;
	}
	else
	{
		value = VALUE + 1.0;
	}
	
	return value;
}

float Core::curve (const float POSITION_START, const float POSITION_CURRENT, const float POSITION_END, const float INTERPOLATION_START, const float INTERPOLATION_END, const float CURVE)
{
	float value = 0;
	
	if ((POSITION_START < POSITION_END && POSITION_CURRENT <= POSITION_START) || (POSITION_START > POSITION_END && POSITION_CURRENT >= POSITION_START))
	{
		value = INTERPOLATION_START;
	}
	else if ((POSITION_START < POSITION_END && POSITION_CURRENT >= POSITION_END) || (POSITION_START > POSITION_END && POSITION_CURRENT <= POSITION_END))
	{
		value = INTERPOLATION_END;
	}
	else
	{
		if (INTERPOLATION_END > INTERPOLATION_START)
		{
			if (CURVE == 0)
			{
				value = INTERPOLATION_START + ((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE > 0)
			{
				value = INTERPOLATION_START + (((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START))) / ((CURVE + 1.0) - (CURVE * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))));
			}
			else
			{
				value = INTERPOLATION_END - (((INTERPOLATION_END - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_END) / (POSITION_START - POSITION_END))) / (1.0 + (-CURVE * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))));
			}
		}
		else
		{
			if (CURVE == 0)
			{
				value = INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)));
			}
			else if (CURVE > 0)
			{
				value = INTERPOLATION_START - (((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START))) / ((CURVE + 1.0) - (CURVE * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))));
			}
			else
			{
				value = INTERPOLATION_END + (((INTERPOLATION_START - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_END) / (POSITION_START - POSITION_END))) / (1.0 + (-CURVE * ((POSITION_CURRENT - POSITION_START) / (POSITION_END - POSITION_START)))));
			}
		}
	}
	
	return value;
}

float Core::wurve (const float POSITION_START, const float POSITION_CURRENT, const float POSITION_END, const float INTERPOLATION_START, const float INTERPOLATION_CENTER, const float INTERPOLATION_END, const float CURVE_START, const float CURVE_END)
{
	const float POSITION_CENTER = POSITION_START + ((POSITION_END - POSITION_START) / 2.0);
	float value = 0;
	
	if ((POSITION_START < POSITION_END && POSITION_CURRENT < POSITION_CENTER) || (POSITION_START > POSITION_END && POSITION_CURRENT > POSITION_CENTER))
	{
		if ((POSITION_START < POSITION_CENTER && POSITION_CURRENT <= POSITION_START) || (POSITION_START > POSITION_CENTER && POSITION_CURRENT >= POSITION_START))
		{
			value = INTERPOLATION_START;
		}
		else if ((POSITION_START < POSITION_CENTER && POSITION_CURRENT >= POSITION_CENTER) || (POSITION_START > POSITION_CENTER && POSITION_CURRENT <= POSITION_CENTER))
		{
			value = INTERPOLATION_CENTER;
		}
		else
		{
			if (INTERPOLATION_CENTER > INTERPOLATION_START)
			{
				if (CURVE_START == 0)
				{
					value = INTERPOLATION_START + ((INTERPOLATION_CENTER - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START)));
				}
				else if (CURVE_START > 0)
				{
					value = INTERPOLATION_CENTER - (((INTERPOLATION_CENTER - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_START - POSITION_CENTER))) / (1.0 + (CURVE_START * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START)))));
				}
				else
				{
					value = INTERPOLATION_START + (((INTERPOLATION_CENTER - INTERPOLATION_START) * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START))) / ((-CURVE_START + 1.0) - (-CURVE_START * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START)))));
				}
			}
			else
			{
				if (CURVE_START == 0)
				{
					value = INTERPOLATION_START - ((INTERPOLATION_START - INTERPOLATION_CENTER) * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START)));
				}
				else if (CURVE_START > 0)
				{
					value = INTERPOLATION_CENTER + (((INTERPOLATION_START - INTERPOLATION_CENTER) * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_START - POSITION_CENTER))) / (1.0 + (CURVE_START * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START)))));
				}
				else
				{
					value = INTERPOLATION_START - (((INTERPOLATION_START - INTERPOLATION_CENTER) * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START))) / ((-CURVE_START + 1.0) - (-CURVE_START * ((POSITION_CURRENT - POSITION_START) / (POSITION_CENTER - POSITION_START)))));
				}
			}
		}
	}
	else if ((POSITION_START < POSITION_END && POSITION_CURRENT > POSITION_CENTER) || (POSITION_START > POSITION_END && POSITION_CURRENT < POSITION_CENTER))
	{
		if ((POSITION_CENTER < POSITION_END && POSITION_CURRENT <= POSITION_CENTER) || (POSITION_CENTER > POSITION_END && POSITION_CURRENT >= POSITION_CENTER))
		{
			value = INTERPOLATION_CENTER;
		}
		else if ((POSITION_CENTER < POSITION_END && POSITION_CURRENT >= POSITION_END) || (POSITION_CENTER > POSITION_END && POSITION_CURRENT <= POSITION_END))
		{
			value = INTERPOLATION_END;
		}
		else
		{
			if (INTERPOLATION_END > INTERPOLATION_CENTER)
			{
				if (CURVE_END == 0)
				{
					value = INTERPOLATION_CENTER + ((INTERPOLATION_END - INTERPOLATION_CENTER) * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER)));
				}
				else if (CURVE_END > 0)
				{
					value = INTERPOLATION_CENTER + (((INTERPOLATION_END - INTERPOLATION_CENTER) * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER))) / ((CURVE_END + 1.0) - (CURVE_END * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER)))));
				}
				else
				{
					value = INTERPOLATION_END - (((INTERPOLATION_END - INTERPOLATION_CENTER) * ((POSITION_CURRENT - POSITION_END) / (POSITION_CENTER - POSITION_END))) / (1.0 + (-CURVE_END * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER)))));
				}
			}
			else
			{
				if (CURVE_END == 0)
				{
					value = INTERPOLATION_CENTER - ((INTERPOLATION_CENTER - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER)));
				}
				else if (CURVE_END > 0)
				{
					value = INTERPOLATION_CENTER - (((INTERPOLATION_CENTER - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER))) / ((CURVE_END + 1.0) - (CURVE_END * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER)))));
				}
				else
				{
					value = INTERPOLATION_END + (((INTERPOLATION_CENTER - INTERPOLATION_END) * ((POSITION_CURRENT - POSITION_END) / (POSITION_CENTER - POSITION_END))) / (1.0 + (-CURVE_END * ((POSITION_CURRENT - POSITION_CENTER) / (POSITION_END - POSITION_CENTER)))));
				}
			}
		}
	}
	else
	{
		value = INTERPOLATION_CENTER;
	}
	
	return value;
}

float Core::range (const float RANGE_START, const float VALUE_CURRENT, const float RANGE_END)
{
	float value = 0;
	
	if (RANGE_START < RANGE_END)
	{
		if (VALUE_CURRENT <= RANGE_START)
		{
			value = RANGE_START;
		}
		else if (VALUE_CURRENT >= RANGE_END)
		{
			value = RANGE_END;
		}
		else
		{
			value = VALUE_CURRENT;
		}
	}
	else if (RANGE_START > RANGE_END)
	{
		if (VALUE_CURRENT <= RANGE_END)
		{
			value = RANGE_END;
		}
		else if (VALUE_CURRENT >= RANGE_START)
		{
			value = RANGE_START;
		}
		else
		{
			value = VALUE_CURRENT;
		}
	}
	else
	{
		value = RANGE_START;
	}
	
	return value;
}

float Core::center (const float VALUE_START, const float VALUE_END)
{
	float value = 0;
	
	if (VALUE_START != VALUE_END)
	{
		value = VALUE_START + ((VALUE_END - VALUE_START) / 2.0);
	}
	else
	{
		value = VALUE_START;
	}
	
	return value;
}

float Core::pow (const float VALUE, const signed long EXPONENT)
{
	signed long iterationPower = 0;
	float value = 0;
	
	if (EXPONENT < 0)
	{
		if (VALUE < 0)
		{
			value = -VALUE;
		}
		else
		{
			value = VALUE;
		}
		
		for (iterationPower = -1; iterationPower > EXPONENT; iterationPower--)
		{
			value *= VALUE;
		}
		
		value = 1.0 / value;
		
		if (VALUE < 0)
		{
			value = -value;
		}
	}
	else if (EXPONENT == 0)
	{
		if (VALUE < 0)
		{
			value = -1;
		}
		else
		{
			value = 1;
		}
	}
	else
	{
		if (VALUE < 0)
		{
			value = -VALUE;
		}
		else
		{
			value = VALUE;
		}
		
		for (iterationPower = 1; iterationPower < EXPONENT; iterationPower++)
		{
			value *= VALUE;
		}
		
		if (VALUE < 0)
		{
			value = -value;
		}
	}
	
	return value;
}

float Core::sqrt (const float RADICAND)
{
	unsigned char iterationAccuracy = 0;
	float searchDecimal = 0.1;
	float foundDecimal = 0;
	bool quickSearch = false;
	unsigned long searchInteger = 1000;
	unsigned int incrementInteger = 1000;
	float sr1 = 0;
	float sr2 = 0;
	float sr3 = 0;
	float value = 0;
	
	if (RADICAND >= 0)
	{
		if (RADICAND == 0)
		{
			value = 0;
		}
		else if (RADICAND < 1)
		{
			while (searchDecimal * searchDecimal <= RADICAND)
			{
				searchDecimal += 0.1;
			}
			
			for (iterationAccuracy = 0; iterationAccuracy < 4; iterationAccuracy++)
			{
				foundDecimal = ((RADICAND / searchDecimal) + searchDecimal) / 2.0;
				searchDecimal = foundDecimal;
			}
			
			value = foundDecimal;
		}
		else if (RADICAND == 1)
		{
			value = 1;
		}
		else
		{
			while (quickSearch == false)
			{
				while (searchInteger * searchInteger <= RADICAND)
				{
					searchInteger += incrementInteger;
				}
				
				searchInteger -= incrementInteger;
				
				if (incrementInteger == 1)
				{
					quickSearch = true;
				}
				else
				{
					incrementInteger /= 10;
				}
			}
			
			sr1 = RADICAND - ((float)(searchInteger) * (float)(searchInteger));
			sr2 = sr1 / (2.0 * (float)(searchInteger));
			sr3 = (float)(searchInteger) + sr2;
			
			value = sr3 - ((sr2 * sr2) / (2.0 * sr3));
		}
	}
	
	return value;
}

unsigned long long Core::fact (const unsigned long INTEGER)
{
	unsigned long iterationFactorial = 1;
	unsigned long long value = 1;
	
	for (iterationFactorial = 1; iterationFactorial <= INTEGER; iterationFactorial++)
	{
		value *= iterationFactorial;
	}
	
	return value;
}

float Core::sin (const float ANGLE)
{
	unsigned long iterationModulo = 0;
	unsigned long foundMultiply = 0;
	bool quickSearch = false;
	float angle = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha5 = 0;
	float sinus = 0;
	
	if (ANGLE >= -180 && ANGLE <= 180)
	{
		angle = ANGLE;
	}
	else if (ANGLE < -180)
	{
		while (quickSearch == false)
		{
			for (iterationModulo = 1; ANGLE + (360.0 * ((float)(foundMultiply) + (float)(iterationModulo))) < -180; iterationModulo *= 10)
			{
			}
			
			if (iterationModulo == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += iterationModulo / 10ul;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE + (360.0 * (float)(foundMultiply));
	}
	else
	{
		while (quickSearch == false)
		{
			for (iterationModulo = 1; ANGLE - (360.0 * ((float)(foundMultiply) + (float)(iterationModulo))) > 180; iterationModulo *= 10)
			{
			}
			
			if (iterationModulo == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += iterationModulo / 10ul;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE - (360.0 * (float)(foundMultiply));
	}
	
	if (angle == -90 || angle == 270)
	{
		sinus = -1;
	}
	else if (angle == -270 || angle == 90)
	{
		sinus = 1;
	}
	else if (angle != -360 && angle != -180 && angle != 0 && angle != 180 && angle != 360)
	{
		alpha = angle * 0.0174532925199432;
		
		alpha2 = alpha * alpha;
		alpha3 = alpha2 * alpha;
		alpha5 = alpha2 * alpha3;
		
		sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
	}
	
	return sinus;
}

float Core::cos (const float ANGLE)
{
	unsigned long iterationModulo = 0;
	unsigned long foundMultiply = 0;
	bool quickSearch = false;
	float angle = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha4 = 0;
	float cosinus = 0;
	
	if (ANGLE >= -180 && ANGLE <= 180)
	{
		angle = ANGLE;
	}
	else if (ANGLE < -180)
	{
		while (quickSearch == false)
		{
			for (iterationModulo = 1; ANGLE + (360.0 * ((float)(foundMultiply) + (float)(iterationModulo))) < -180; iterationModulo *= 10)
			{
			}
			
			if (iterationModulo == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += iterationModulo / 10ul;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE + (360.0 * (float)(foundMultiply));
	}
	else
	{
		while (quickSearch == false)
		{
			for (iterationModulo = 1; ANGLE - (360.0 * ((float)(foundMultiply) + (float)(iterationModulo))) > 180; iterationModulo *= 10)
			{
			}
			
			if (iterationModulo == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += iterationModulo / 10ul;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE - (360.0 * (float)(foundMultiply));
	}
	
	if (angle == -180 || angle == 180)
	{
		cosinus = -1;
	}
	else if (angle == -360 || angle == 0 || angle == 360)
	{
		cosinus = 1;
	}
	else if (angle != -270 && angle != -90 && angle != 90 && angle != 270)
	{
		alpha = angle * 0.0174532925199432;
		
		alpha2 = alpha * alpha;
		alpha4 = alpha2 * alpha2;
		
		cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
	}
	
	return cosinus;
}

float Core::tan (const float ANGLE)
{
	unsigned long iterationModulo = 0;
	unsigned long foundMultiply = 0;
	bool quickSearch = false;
	float angle = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha5 = 0;
	float tangent = 0;
	
	if (ANGLE >= -90 && ANGLE <= 90)
	{
		angle = ANGLE;
	}
	else if (ANGLE < -90)
	{
		while (quickSearch == false)
		{
			for (iterationModulo = 1; ANGLE + (180.0 * ((float)(foundMultiply) + (float)(iterationModulo))) < -90; iterationModulo *= 10)
			{
			}
			
			if (iterationModulo == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += iterationModulo / 10ul;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE + (180.0 * (float)(foundMultiply));
	}
	else
	{
		while (quickSearch == false)
		{
			for (iterationModulo = 1; ANGLE - (180.0 * ((float)(foundMultiply) + (float)(iterationModulo))) > 90; iterationModulo *= 10)
			{
			}
			
			if (iterationModulo == 1)
			{
				quickSearch = true;
			}
			else
			{
				foundMultiply += iterationModulo / 10ul;
			}
		}
		
		foundMultiply++;
		
		angle = ANGLE - (180.0 * (float)(foundMultiply));
	}
	
	if (angle == -45)
	{
		tangent = -1;
	}
	else if (angle == -90 || angle == 0 || angle == 90)
	{
		tangent = 0;
	}
	else if (angle == 45)
	{
		tangent = 1;
	}
	else
	{
		if (angle > -45 && angle < 45)
		{
			alpha = angle * 0.0174532925199432;
		}
		else
		{
			if (angle < 0)
			{
				alpha = (-90.0 - angle) * 0.0174532925199432;
			}
			else
			{
				alpha = (90.0 - angle) * 0.0174532925199432;
			}
		}
		
		alpha2 = alpha * alpha;
		alpha3 = alpha2 * alpha;
		alpha5 = alpha2 * alpha3;
		
		if (angle > -45 && angle < 45)
		{
			tangent = alpha + ((alpha3 / 3.0) + ((2.0 * alpha5) / 15.0) + ((17.0 * (alpha5 * alpha2)) / 315.0));
		}
		else
		{
			tangent = 1.0 / (alpha + ((alpha3 / 3.0) + ((2.0 * alpha5) / 15.0) + ((17.0 * (alpha5 * alpha2)) / 315.0)));
		}
	}
	
	return tangent;
}

float Core::arcsin (const float SINUS)
{
	float angle = 0;
	float anglePrevious = 0;
	float sinus = 0;
	float sinusPositive = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha5 = 0;
	float incrementAngle = 10;
	unsigned char decimal = 0;
	unsigned char error = 0;
	bool quickSearch = false;
	
	if (SINUS <= -1)
	{
		angle = -90;
	}
	else if (SINUS == 0)
	{
		angle = 0;
	}
	else if (SINUS >= 1)
	{
		angle = 90;
	}
	else
	{
		if (SINUS < 0)
		{
			sinusPositive = -SINUS;
		}
		else
		{
			sinusPositive = SINUS;
		}
		
		while (quickSearch == false)
		{
			alpha = angle * 0.0174532925199432;
			
			alpha2 = alpha * alpha;
			alpha3 = alpha2 * alpha;
			alpha5 = alpha2 * alpha3;
			
			sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
			
			if (sinus < sinusPositive && incrementAngle >= 1)
			{
				anglePrevious = angle;
				angle += incrementAngle;
			}
			else if (sinus < sinusPositive && incrementAngle < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += incrementAngle;
				
				error++;
			}
			else if (sinus == sinusPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				incrementAngle /= 10;
				angle = anglePrevious + incrementAngle;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (SINUS < 0)
		{
			angle = -angle;
		}
	}
	
	return angle;
}

float Core::arccos (const float COSINUS)
{
	float angle = 0;
	float anglePrevious = 0;
	float cosinus = 0;
	float cosinusPositive = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha4 = 0;
	float incrementAngle = 10;
	unsigned char decimal = 0;
	unsigned char error = 0;
	bool quickSearch = false;
	
	if (COSINUS <= -1)
	{
		angle = 180;
	}
	else if (COSINUS == 0)
	{
		angle = 90;
	}
	else if (COSINUS >= 1)
	{
		angle = 0;
	}
	else
	{
		if (COSINUS < 0)
		{
			cosinusPositive = -COSINUS;
		}
		else
		{
			cosinusPositive = COSINUS;
		}
		
		while (quickSearch == false)
		{
			alpha = angle * 0.0174532925199432;
			
			alpha2 = alpha * alpha;
			alpha4 = alpha2 * alpha2;
			
			cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
			
			if (cosinus > cosinusPositive && incrementAngle >= 1)
			{
				anglePrevious = angle;
				angle += incrementAngle;
			}
			else if (cosinus > cosinusPositive && incrementAngle < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += incrementAngle;
				
				error++;
			}
			else if (cosinus == cosinusPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				incrementAngle /= 10;
				angle = anglePrevious + incrementAngle;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (COSINUS < 0)
		{
			angle = 180.0 - angle;
		}
	}
	
	return angle;
}

float Core::arctan (const float TANGENT)
{
	float angle = 0;
	float anglePrevious = 0;
	float sinus = 0;
	float cosinus = 0;
	float tangent = 0;
	float tangentPositive = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha4 = 0;
	float alpha5 = 0;
	float incrementAngle = 10;
	unsigned char decimal = 0;
	unsigned char error = 0;
	bool quickSearch = false;
	
	if (TANGENT <= -9999999999999999)
	{
		angle = -90;
	}
	else if (TANGENT == -1)
	{
		angle = -45;
	}
	else if (TANGENT == 0)
	{
		angle = 0;
	}
	else if (TANGENT == 1)
	{
		angle = 45;
	}
	else if (TANGENT >= 9999999999999999)
	{
		angle = 90;
	}
	else
	{
		if (TANGENT < 0)
		{
			tangentPositive = -TANGENT;
		}
		else
		{
			tangentPositive = TANGENT;
		}
		
		while (quickSearch == false)
		{
			alpha = angle * 0.0174532925199432;
			
			alpha2 = alpha * alpha;
			alpha3 = alpha2 * alpha;
			alpha4 = alpha2 * alpha2;
			alpha5 = alpha2 * alpha3;
			
			sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
			cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
			tangent = sinus / cosinus;
			
			if (tangent < tangentPositive && incrementAngle >= 1)
			{
				anglePrevious = angle;
				angle += incrementAngle;
			}
			else if (tangent < tangentPositive && incrementAngle < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += incrementAngle;
				
				error++;
			}
			else if (tangent == tangentPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				incrementAngle /= 10;
				angle = anglePrevious + incrementAngle;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (TANGENT < 0)
		{
			angle = -angle;
		}
	}
	
	return angle;
}

float Core::arctan2 (const float X, const float Y)
{
	const float TANGENT = Y / X;
	float angle = 0;
	float anglePrevious = 0;
	float sinus = 0;
	float cosinus = 0;
	float tangent = 0;
	float tangentPositive = 0;
	float alpha = 0;
	float alpha2 = 0;
	float alpha3 = 0;
	float alpha4 = 0;
	float alpha5 = 0;
	float incrementAngle = 10;
	unsigned char decimal = 0;
	unsigned char error = 0;
	bool quickSearch = false;
	
	if (TANGENT <= -9999999999999999)
	{
		angle = -90;
	}
	else if (TANGENT == -1)
	{
		angle = -45;
	}
	else if (TANGENT == 0)
	{
		angle = 0;
	}
	else if (TANGENT == 1)
	{
		angle = 45;
	}
	else if (TANGENT >= 9999999999999999)
	{
		angle = 90;
	}
	else
	{
		if (TANGENT < 0)
		{
			tangentPositive = -TANGENT;
		}
		else
		{
			tangentPositive = TANGENT;
		}
		
		while (quickSearch == false)
		{
			alpha = angle * 0.0174532925199432;
			
			alpha2 = alpha * alpha;
			alpha3 = alpha2 * alpha;
			alpha4 = alpha2 * alpha2;
			alpha5 = alpha2 * alpha3;
			
			sinus = alpha - (alpha3 / 6.0) + (alpha5 / 120.0) - ((alpha2 * alpha5) / 5040.0);
			cosinus = 1.0 - ((alpha2 / 2.0) - (alpha4 / 24.0) + ((alpha2 * alpha4) / 720.0) - ((alpha4 * alpha4) / 40320.0));
			tangent = sinus / cosinus;
			
			if (tangent < tangentPositive && incrementAngle >= 1)
			{
				anglePrevious = angle;
				angle += incrementAngle;
			}
			else if (tangent < tangentPositive && incrementAngle < 1 && error < 9)
			{
				anglePrevious = angle;
				angle += incrementAngle;
				
				error++;
			}
			else if (tangent == tangentPositive || decimal == 4 + 1)
			{
				quickSearch = true;
			}
			else
			{
				incrementAngle /= 10;
				angle = anglePrevious + incrementAngle;
				
				decimal++;
				error = 0;
			}
			
			if (angle > 90.0001)
			{
				angle = 89.9999;
				quickSearch = true;
			}
		}
		
		if (TANGENT < 0)
		{
			angle = -angle;
		}
	}
	
	if (X < 0)
	{
		if (Y >= 0)
		{
			angle += 180;
		}
		else
		{
			angle -= 180;
		}
	}
	
	return angle;
}
